# -----------------------------
# Sous-fonctions
# -----------------------------

def f_varlit(VarLit: int) -> int:
    """
    F_VarLit – Variabilité de la largeur du lit

    Calcule un score en fonction de la variabilité de la largeur du lit donnée par VarLit.

    Paramètres :
        VarLit (int) : Code de variabilité de la largeur du lit
            - 0 → non défini (exclusion)
            - 1 → variabilité marquée
            - 2 → variabilité restreinte
            - autre → aucune variabilité

    Retour :
        int : Score F_VarLit

    Exemples :
        f_varlit(0) → -10
        f_varlit(1) → 0
        f_varlit(2) → 2
        f_varlit(5) → 3
    """

    # Normalisation des valeurs None
    if VarLit == None:
        VarLit = 0

    # Attribution du score
    if VarLit == 0:
        return -10
    elif VarLit == 1:
        return 0
    elif VarLit == 2:
        return 2
    else:
        return 3

def f_amlit(AmLit: int, MatLit: int) -> int:
    """
    F_AmLit – Aménagement du fond du lit (Sohle)

    Calcule un score lié à l’aménagement du fond du lit en fonction de 
    AmLit (intensité d’aménagement) et MatLit (type de matériau).

    Paramètres :
        AmLit (int) : Niveau d’aménagement du fond de lit
            - 0 → non défini (exclusion)
            - 1 → 0 % aménagé
            - 2 → < 10 % aménagé
            - 3 → 10–30 % aménagé
            - >3 → >30 % aménagé
        MatLit (int) : Matériau du lit
            - 0 → non défini
            - 1 → perméable
            - autres → étanche

    Retour :
        int : Score F_AmLit, ou -10 pour exclusion

    Exemples :
        f_amlit(0, 1) → -10
        f_amlit(2, 3) → 1
        f_amlit(4, 1) → 2
        f_amlit(4, 5) → 3
    """

    # Normalisation des valeurs None
    if AmLit == None:
        AmLit = 0
    if MatLit == None:
        MatLit = 0

    # Attribution du score
    if AmLit == 0 or (AmLit > 3 and MatLit == 0):
        return -10
    elif AmLit == 1:
        return 0
    elif AmLit == 2:
        return 1
    elif AmLit == 3:
        return 2
    elif AmLit > 3:
        return 2 if MatLit == 1 else 3
    else:
        return -10  # sécurité

def f_buk(RenfBergeG: int, MatBergeG: int, RenfBergeD: int, MatBergeD: int) -> int:
    """
    F_Buk – Renforcement du pied de berge

    Calcule un score de renforcement du pied de berge en fonction des
    valeurs de RenfBergeG, MatBergeG, RenfBergeD et MatBergeD.

    Règles principales :
    - Si l’un des codes de renforcement = 0 → -10 (exclusion).
    - Si renforcement ≠ 1 et matériau = 0 → -10 (incohérence).
    - Sinon, un score partiel est calculé pour chaque berge :
        * Renforcement 1–2 → 0
        * Renforcement 3 → 0.5 (perméable) ou 1 (étanche)
        * Renforcement 4 → 1.5 (perméable) ou 2 (étanche)
        * Renforcement >4 → 2.5 (perméable) ou 3 (étanche)
    - Score final = moyenne des deux berges, arrondie à l’entier le plus proche
      (0.5 arrondi vers le haut).

    Retour :
        Un entier représentant la note F_Buk.
    """

    # Remplacer les valeurs None par 0
    RenfBergeG = 0 if RenfBergeG == None else RenfBergeG
    MatBergeG  = 0 if MatBergeG == None else MatBergeG
    RenfBergeD = 0 if RenfBergeD == None else RenfBergeD
    MatBergeD = 0 if MatBergeD == None else MatBergeD

    # Cas d'exclusion
    if RenfBergeG == 0 or RenfBergeD == 0:
        return -10
    if (RenfBergeG != 1 and MatBergeG == 0) or (RenfBergeD != 1 and MatBergeD == 0):
        return -10

    # ---- Côté gauche ----
    if RenfBergeG in (1, 2):
        LBuk = 0
    elif RenfBergeG == 3:
        LBuk = 0.5 if 1 <= MatBergeG <= 3 else 1
    elif RenfBergeG == 4:
        LBuk = 1.5 if 1 <= MatBergeG <= 3 else 2
    else:
        LBuk = 2.5 if 1 <= MatBergeG <= 3 else 3

    # ---- Côté droit ----
    if RenfBergeD in (1, 2):
        RBuk = 0
    elif RenfBergeD == 3:
        RBuk = 0.5 if 1 <= MatBergeD <= 3 else 1
    elif RenfBergeD == 4:
        RBuk = 1.5 if 1 <= MatBergeD <= 3 else 2
    else:
        RBuk = 2.5 if 1 <= MatBergeD <= 3 else 3

    # ---- Moyenne et arrondi ----
    Punkte = (LBuk + RBuk) / 2

    return int(Punkte + 0.5)  # arrondi "0.5 vers le haut"

def f_rive(LargeMoyLit, VarLit, LargeRiveG, LargeRiveD, NatureRiveG, NatureRiveD):
    """
    F_Rive – Évaluation des rives

    Calcule un score des rives à partir des largeurs moyennes du lit, de la variabilité,
    des largeurs de rive gauche et droite, et de la nature des rives.

    Paramètres :
        LargeMoyLit (float) : Largeur moyenne du lit
        VarLit (int) : Variabilité du lit (1,2,3)
        LargeRiveG (float) : Largeur de la rive gauche
        LargeRiveD (float) : Largeur de la rive droite
        NatureRiveG (int) : Nature rive gauche (1=géré,2=faux,3=aucun)
        NatureRiveD (int) : Nature rive droite (1=géré,2=faux,3=aucun)

    Retour :
        int : Score final arrondi à l'entier le plus proche (0.5 vers le haut)
              ou -10 pour exclusion.
    """

    # Remplacer les valeurs None par 0
    VarLit = 0 if VarLit == None else VarLit
    NatureRiveG = 0 if NatureRiveG == None else NatureRiveG
    NatureRiveD = 0 if NatureRiveD == None else NatureRiveD

    # --- Calcul rive gauche ---
    if LargeRiveG == None or LargeMoyLit == None or LargeMoyLit == 0 or VarLit == 0:
        P_LUferBer = 0
    elif LargeRiveG == 0:
        P_LUferBer = 3
    else:
        if ((VarLit == 1 and ((LargeMoyLit < 2 and LargeRiveG >= 5) 
            or (2 <= LargeMoyLit <= 15 and LargeRiveG >= 10/13*LargeMoyLit + 45/13)
            or (LargeMoyLit > 15 and LargeRiveG >= 15)))
            or (VarLit == 2 and ((LargeMoyLit < 4/3 and LargeRiveG >= 5)
            or (4/3 <= LargeMoyLit <= 10 and LargeRiveG >= 15/13*LargeMoyLit + 45/13)
            or (LargeMoyLit > 10 and LargeRiveG >= 15)))
            or (VarLit == 3 and ((LargeMoyLit < 1 and LargeRiveG >= 5)
            or (1 <= LargeMoyLit <= 15/2 and LargeRiveG >= 20/13*LargeMoyLit + 45/13)
            or (LargeMoyLit > 15/2 and LargeRiveG >= 15)))):
            P_LUferBer = 1
        else:
            P_LUferBer = 2

    # --- Calcul rive droite ---
    if LargeRiveD == None or LargeMoyLit == None or LargeMoyLit == 0 or VarLit == 0:
        P_RUferBer = 0
    elif LargeRiveD == 0:
        P_RUferBer = 3
    else:
        if ((VarLit == 1 and ((LargeMoyLit < 2 and LargeRiveD >= 5) 
            or (2 <= LargeMoyLit <= 15 and LargeRiveD >= 10/13*LargeMoyLit + 45/13)
            or (LargeMoyLit > 15 and LargeRiveD >= 15)))
            or (VarLit == 2 and ((LargeMoyLit < 4/3 and LargeRiveD >= 5)
            or (4/3 <= LargeMoyLit <= 10 and LargeRiveD >= 15/13*LargeMoyLit + 45/13)
            or (LargeMoyLit > 10 and LargeRiveD >= 15)))
            or (VarLit == 3 and ((LargeMoyLit < 1 and LargeRiveD >= 5)
            or (1 <= LargeMoyLit <= 15/2 and LargeRiveD >= 20/13*LargeMoyLit + 45/13)
            or (LargeMoyLit > 15/2 and LargeRiveD >= 15)))):
            P_RUferBer = 1
        else:
            P_RUferBer = 2

    # --- Fonction interne pour calculer le score en fonction de P et Nature ---
    def score_ber(P, Nature):
        if P == 3 or Nature == 3:
            return 3
        elif P == 1:
            return 0 if Nature == 1 else 1.5
        else:  # P == 2
            return 2 if Nature == 1 else 3

    # --- Calcul score final avec cas d'exclusion ---
    if P_LUferBer == 0 or P_RUferBer == 0:
        Punkte = -10
    elif (P_LUferBer < 3 and NatureRiveG == 0) or (P_RUferBer < 3 and NatureRiveD == 0):
        Punkte = -10
    else:
        LUfer = score_ber(P_LUferBer, NatureRiveG)
        RUfer = score_ber(P_RUferBer, NatureRiveD)
        Punkte = (LUfer + RUfer) / 2
        Punkte = int(Punkte + 0.5)  # arrondi 0.5 vers le haut

    return Punkte

# -----------------------------
# Fonction principale pour calculer la note finale
# -----------------------------
def f_note(ssTerre,
                         VarLit, AmLit, MatLit,
                         RenfBergeG, MatBergeG, RenfBergeD, MatBergeD,
                         LargeMoyLit, LargeRiveG, LargeRiveD, NatureRiveG, NatureRiveD):
    """
    Calcule directement la note finale d’un tronçon à partir des valeurs brutes.
    """
    P_BVar = f_varlit(VarLit)
    #print(P_BVar)
    P_Sohle = f_amlit(AmLit, MatLit)
    #print(P_Sohle)
    P_Buk = f_buk(RenfBergeG, MatBergeG, RenfBergeD, MatBergeD)
    #print(P_Buk)
    P_Ufer = f_rive(LargeMoyLit, VarLit, LargeRiveG, LargeRiveD, NatureRiveG, NatureRiveD)
    #print("rive:" + str(P_Ufer))

    # Calcul de la note finale
    P_Eindol = 0 if ssTerre == 0 else -100
    #print(P_Eindol)
    Punkte_Sum = P_Eindol + P_BVar + P_Sohle + P_Buk + P_Ufer
    #print(Punkte_Sum)

    if Punkte_Sum <= -100:
        return 5
    elif -100 < Punkte_Sum < 0:
        return 0
    else:
        return int(Punkte_Sum / 4 + 1 + 0.5)